//
//  Service.mm
//  Handy Harp Client
//  
//  Created by KIRA Ryouta on 2025/11/15.
//
/*
 MIT License

 Copyright (c) 2025 PROCYON STUDIO CO., LTD.
 Copyright (c) 2025 KIRA Ryouta

 Permission is hereby granted, free of charge, to any person obtaining a copy of this software and associated documentation files (the "Software"), to deal in the Software without restriction, including without limitation the rights to use, copy, modify, merge, publish, distribute, sublicense, and/or sell copies of the Software, and to permit persons to whom the Software is furnished to do so, subject to the following conditions:

 The above copyright notice and this permission notice shall be included in all copies or substantial portions of the Software.

 THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.
 */

#import "Service.h"
#import <atomic>
#import <vector>

namespace HHC {

struct Connection {
    NSNetService* service;
    NSInputStream* iStream;
    NSOutputStream* oStream;
};

}

@interface HHC_NetworkClient () <NSNetServiceBrowserDelegate, NSNetServiceDelegate, NSStreamDelegate>

@end

@implementation HHC_NetworkClient
{
    NSTimer* dummyTimer;
    NSTimer* reconnectTimer;
    NSNetServiceBrowser* browser;

    std::vector<HHC::Connection> servers;

    NSRunLoop* runloop;
    NSTimer* outputTimer;
}

- (id)init
{
    if (self = [super init]) {
        __weak HHC_NetworkClient* weakSelf = self;
        dispatch_async(dispatch_get_global_queue(DISPATCH_QUEUE_PRIORITY_DEFAULT, 0), ^{
            NSRunLoop* runloop = [NSRunLoop currentRunLoop];
            @autoreleasepool {
                HHC_NetworkClient* self = weakSelf;
                if (self) {
                    self->runloop = runloop;
                    self->dummyTimer = [NSTimer timerWithTimeInterval:86400. repeats:YES block:^(NSTimer * _Nonnull timer) {}];
                    [runloop addTimer:self->dummyTimer forMode:NSDefaultRunLoopMode];
                }
            }
            while (1) {
                @autoreleasepool {
                    HHC_NetworkClient* self = weakSelf;
                    if (!self || !self->dummyTimer) {
                        break;
                    }
                }
                [runloop runUntilDate:[NSDate distantFuture]];
            }
        });

        browser = [[NSNetServiceBrowser alloc] init];
        browser.delegate = self;
        [self requestService];
        _running = YES;
    }
    return self;
}

- (void)dealloc
{
    [self stopAll];
    [dummyTimer invalidate];
    dummyTimer = nil;
    NSTimer* timer = [NSTimer timerWithTimeInterval:0. repeats:NO block:^(NSTimer * _Nonnull timer) {}];
    [runloop addTimer:timer forMode:NSRunLoopCommonModes];
}

- (void)requestService
{
    [browser searchForServicesOfType:@"_handyharp2._tcp" inDomain:@""];
}

- (void)netServiceBrowser:(NSNetServiceBrowser *)browser didFindService:(NSNetService *)service moreComing:(BOOL)moreComing
{
#if DEBUG
    NSLog(@"didFindService: %@", service);
#endif
    service.delegate = self;
    [service resolveWithTimeout:5.];
    NSTimer* timer = [NSTimer timerWithTimeInterval:0. repeats:NO block:^(NSTimer * _Nonnull timer) {
        self->servers.emplace_back(service);
    }];
    [runloop addTimer:timer forMode:NSRunLoopCommonModes];
}

- (void)disconnectService:(NSNetService*)s
{
    for (auto iter = servers.begin(); iter != servers.end(); ++iter) {
        auto& service = iter->service;
        if (service == s) {
            [iter->iStream close];
            [iter->oStream close];
            [service stop];
            servers.erase(iter);
            [self checkConnected];
            break;
        }
    }
}

- (void)netServiceBrowser:(NSNetServiceBrowser *)browser didRemoveService:(NSNetService *)s moreComing:(BOOL)moreComing
{
#if DEBUG
    NSLog(@"didRemoveService: %@", s);
#endif
    NSTimer* timer = [NSTimer timerWithTimeInterval:0. repeats:NO block:^(NSTimer * _Nonnull timer) {
        [self disconnectService:s];
    }];
    [runloop addTimer:timer forMode:NSRunLoopCommonModes];
}

- (void)netService:(NSNetService *)s didNotResolve:(NSDictionary<NSString *,NSNumber *> *)errorDict
{
#if DEBUG
    NSLog(@"didNotResolve: %@, error:%@", s, errorDict);
#endif
    NSTimer* timer = [NSTimer timerWithTimeInterval:0. repeats:NO block:^(NSTimer * _Nonnull timer) {
        [self disconnectService:s];
    }];
    [runloop addTimer:timer forMode:NSRunLoopCommonModes];
}

- (void)netServiceDidResolveAddress:(NSNetService *)sender
{
#if DEBUG
    NSLog(@"didResolveAddress: %@", sender);
#endif
    NSTimer* timer = [NSTimer timerWithTimeInterval:0. repeats:NO block:^(NSTimer * _Nonnull timer) {
        NSInputStream* is = nil;
        NSOutputStream* os = nil;
        if (![sender getInputStream:&is outputStream:&os]) {
            NSLog(@"Cannot get streams.");
            [self disconnectService:sender];
            return;
        }
#pragma clang optimize off
        while (!self->runloop) {
            usleep(100);
        }
#pragma clang optimize on

        for (auto& conn : self->servers) {
            if (conn.service == sender) {
                conn.iStream = is;
                conn.oStream = os;

                is.delegate = self;
                [is scheduleInRunLoop:self->runloop forMode:NSRunLoopCommonModes];
                [is open];

                [os scheduleInRunLoop:self->runloop forMode:NSRunLoopCommonModes];
                [os open];

                [self checkConnected];
                break;
            }
        }
        if (!self->outputTimer) {
            __weak HHC_NetworkClient* weakSelf = self;
            self->outputTimer = [NSTimer timerWithTimeInterval:60. repeats:YES block:^(NSTimer * _Nonnull timer) {
                HHC_NetworkClient* self = weakSelf;
                if (self) {
                    for (auto& conn : self->servers) {
                        uint8_t byte1 = 0;
                        [conn.oStream write:&byte1 maxLength:1];
                    }
                }
            }];
            [self->runloop addTimer:self->outputTimer forMode:NSRunLoopCommonModes];
        }
    }];
    [runloop addTimer:timer forMode:NSRunLoopCommonModes];
}

- (void)stream:(NSStream *)aStream handleEvent:(NSStreamEvent)eventCode
{
    if ([aStream isKindOfClass:[NSInputStream class]]) {
        __unsafe_unretained NSInputStream* iStream = (NSInputStream*)aStream;
        bool disConn = false;
        if (eventCode & (NSStreamEventEndEncountered | NSStreamEventErrorOccurred)) {
            disConn = true;
        }
        if (eventCode & NSStreamEventHasBytesAvailable) {
            uint32_t buf;
            uint64_t received = 0;
            while (iStream.hasBytesAvailable) {
                auto length = [iStream read:(uint8_t*)&buf maxLength:4];
                if (length < 0) {
                    received = 0;
                    break;
                }
                if (length == 0) {
                    break;
                }
                received += length;
                dispatch_async(dispatch_get_main_queue(), ^{
                    if (self.receiveCommand) {
                        self.receiveCommand(buf);
                    }
                });
            }
            if (received == 0) {
                disConn = true;
            }
        }
        if (disConn) {
            for (auto iter = servers.begin(); iter != servers.end(); ++iter) {
                if (iter->iStream == iStream) {
                    [iter->iStream close];
                    iter->iStream = nil;
                    [iter->oStream close];
                    iter->oStream = nil;
                    [iter->service stop];
                    iter->service = nil;
                    servers.erase(iter);
                    break;
                }
            }
            [self checkConnected];
        }
    }
}

- (void)stopAll
{
    [self->outputTimer invalidate];
    self->outputTimer = nil;
    [self->browser stop];
    for (auto& conn : self->servers) {
        [conn.iStream close];
        [conn.oStream close];
        [conn.service stop];
    }
    self->servers.clear();
}

- (void)setRunning:(BOOL)running
{
    if (_running != running) {
        _running = running;

        NSTimer* timer = [NSTimer timerWithTimeInterval:0. repeats:NO block:^(NSTimer * _Nonnull timer) {
            if (running) {
                [self requestService];
            }
            else {
                [self stopAll];
                [self checkConnected];
            }
        }];
        [runloop addTimer:timer forMode:NSRunLoopCommonModes];
    }
}

- (void)checkConnected
{
    bool isConnected = false;
    for (auto& conn : servers) {
        if (conn.iStream && conn.oStream) {
            isConnected = true;
            break;
        }
    }
    dispatch_async(dispatch_get_main_queue(), ^{
        if (self.connected != isConnected) {
            self.connected = isConnected;
        }
    });
}

@end
